import json
from pathlib import Path
from typing import Optional

from prettytable import PrettyTable


# Function to read and extract data from JSON file
def read_json_file(file_path: Path) -> Optional[dict]:
    """
    This function is used to read and extract data from a JSON file.
    """
    try:
        if not file_path.exists():  # Check if file exists using Path methods
            print(f"File does not exist: {file_path}")
            return None
        with file_path.open("r") as file:
            return json.load(file)
    except PermissionError:
        print(f"Permission denied: {file_path}")
    except json.JSONDecodeError:
        print(f"Invalid JSON format in file: {file_path}")
    except Exception as e:
        print(f"An error occurred while reading {file_path}: {e}")
    return None


def ultrastack_display_account():
    """
    This function is used to display the UltraStack
    account information in a table format.
    """
    # Path to the JSON files
    json_variables_path = Path("/root/.ansible/logs/wp3_run/latest/variables")
    json_branches_path = Path("/root/.ansible/logs/wp3_run/latest/branches")

    # Read and extract specific data from the JSON files
    variables_data = read_json_file(json_variables_path)

    # Read and extract specific data from the JSON files
    branch_data = read_json_file(json_branches_path)

    # Create account Table
    table = PrettyTable(["1", "2"])
    table.align["1"] = "r"
    table.align["2"] = "l"

    # Disable Table Header
    table.header = False

    # Check if data was successfully extracted from variables file
    if variables_data is not None:
        # Hostname
        hostname = variables_data.get("ansible_facts", {}).get("hostname", "")
        table.add_row(["VPS", hostname])

        # Determine plan name from system memory
        mem_mb = int(variables_data.get("ansible_memtotal_mb", ""))
        if mem_mb < 1024:
            mem_string = f"{mem_mb}MB"
        else:
            mem_string = f"{int(mem_mb / 1024)}GB"
        table.add_row(["Plan", f"UltraStack {mem_string} RAM"])

        # IP address
        ip = variables_data.get("inventory_hostname", "")
        table.add_row(["IP", ip])

        # Domain
        domain = variables_data.get("site_domain", "")
        table.add_row(["Domain", domain])

        # E-mail
        email = variables_data.get("site_email", "")
        table.add_row(["E-mail", email])

        # Nginx Cache Profile
        profile = variables_data.get("nginx_cache_profile", "")
        table.add_row(["NGINX Cache Profile", profile])

        # PHP Version
        php = variables_data.get("php_version", "")
        table.add_row(["PHP Version", php])

    # Check if data was successfully extracted from branches file
    if branch_data is not None:
        # Build table rows
        playbook = branch_data.get("wp3-ultrastack-playbook", {}).get("branch")
        table.add_row(["Playbook Version", playbook])

    # Print table
    print(table)
