#!/opt/imh-python/bin/python3
"""Removes Boldgrid backups older than 7 days"""
from collections import defaultdict
import re
from pathlib import Path
from os.path import getctime
import rads
from guds_modules.base import ModuleBase

# The directory is usually ~/boldgrid_backup/, but if it's not writable, it
# will go in the site's root dir with a random name, so search for the
# filename pattern instead: boldgrid-backup-${SITE_ID}-${CRC32}-${DATE}.zip
FIND_REGEX = r'.*/boldgrid\-backup\-.+\-[0-9a-f]{8}\-[0-9]{8}\-[0-9]{6}\.zip$'
PY_RE = re.compile(
    r'boldgrid\-backup\-(.+)\-[0-9a-f]{8}\-[0-9]{8}\-[0-9]{6}\.zip$'
)


class Module(ModuleBase):
    """Removes Boldgrid backups older than 7 days"""

    def run_module(self, homedir):
        """This function locates boldgrid backups and handles their deletion"""
        # List of backups over 7 days old
        backup_files = self.find(
            homedir,
            type='f',
            regex=FIND_REGEX,
            mtime="+7",
        )
        if not backup_files:
            return {}  # no email
        # if total size is < 10GiB, do nothing
        if sum(map(self.calc_bytes, backup_files)) < 10*2**30:
            self.logger.debug('backups too small, skipping %s', homedir)
            return {}  # no email
        # group by parent directories
        parents: defaultdict[Path, list[Path]] = defaultdict(list)
        for backup in backup_files:
            parents[backup.parent].append(backup)
        # within each parent dir, group by site_id
        for dir_files in parents.values():
            sites: defaultdict[str, list[Path]] = defaultdict(list)
            for path in dir_files:
                site_id: str = PY_RE.match(str(path.name)).group(1)
                sites[site_id].append(path)
            # remove all but the newest for each site
            for site_files in sites.values():
                if len(site_files) >= 2:
                    site_files.remove(max(site_files, key=getctime))
                    self.delete_items(site_files)
        if rads.IMH_CLASS == 'reseller':
            return {}  # no email
        return {
            'template': 379 if rads.IMH_CLASS == 'hub' else 104,
            'variable_1': ', '.join(map(str, parents)),
        }
