#!/bin/bash
# Functions
# Authored By: Corey Stemen (coreys@inmotionhosting.com)

# Global Variables
NGINX_INSTALLED='' # If NGiNX is installed this will get set
DOCUMENT_ROOT='' # Holds doc root for target domain
SUBVERSION=0 # Holds the sub-version integer for Apache
USER='' # Holds user owner for target domain
DOMAIN='' # Holds the target domain - supplied via $1 or entered manually via 'read'

function lock_domain {
  echo "Document Root: $DOCUMENT_ROOT"
  echo "Apache Version: 2.$SUBVERSION"
  if [[ ! -z "$NGINX_INSTALLED" ]]; then
    echo $NGINX_INSTALLED
  fi
  chattr -i $DOCUMENT_ROOT/.htaccess 2>/dev/null #Just remove immutable outright, this covers instances where it was set immutable and not removed, or was immutable at cx request, etc
  touch $DOCUMENT_ROOT/.htaccess 2>/dev/null #test editing
  if [[ ! $? -eq 0 ]]; then
    echo 'Cannot Edit .htaccess'
    exit 1
  fi
  printf '.htaccess found...Editing...\n'
  printf '#########################################################################\n# Website has been disabled by legal department #\n# Please reach out to legal@inmotionhosting.com #\n#########################################################################\n' >> $DOCUMENT_ROOT/.htaccess_legal
  if [[ $SUBVERSION -eq 2 ]]; then
    echo "deny from all" >> $DOCUMENT_ROOT/.htaccess_legal
  else
    echo "Require all denied" >> $DOCUMENT_ROOT/.htaccess_legal
  fi
  cat $DOCUMENT_ROOT/.htaccess_legal $DOCUMENT_ROOT/.htaccess >> $DOCUMENT_ROOT/.htaccess_combined && mv $DOCUMENT_ROOT/.htaccess{,LegalDisabled} && mv $DOCUMENT_ROOT/.htaccess_combined $DOCUMENT_ROOT/.htaccess
  chattr +i $DOCUMENT_ROOT/.htaccess #Lock down new .htaccess
  rm -f $DOCUMENT_ROOT/.htaccess_legal #no longer need this
  echo 'Done!'
}

function unlock_domain {
  chattr -i $DOCUMENT_ROOT/.htaccess 2>/dev/null #.htaccess should exist and should be immutable but redirect stderr just in case as we don't need to see errors from this
  if [[ -f $DOCUMENT_ROOT/.htaccessLegalDisabled ]]; then #If it was disabled by this script then this file should exist and easiest reversion is to just put it back
    rm -f $DOCUMENT_ROOT/.htaccess && mv -v $DOCUMENT_ROOT/.htaccessLegalDisabled $DOCUMENT_ROOT/.htaccess
  else
    cat $DOCUMENT_ROOT/.htaccess | sed 1,5d >> $DOCUMENT_ROOT/.htaccess_new && mv -f $DOCUMENT_ROOT/.htaccess{,Disabled} && mv -f $DOCUMENT_ROOT/.htaccess_new $DOCUMENT_ROOT/.htaccess #Not removing .htaccessDisabled here just for redundancy in case of inadvertent corruption or something and/or if the following check fails
  fi
  unset check && check=$(grep -i 'disabled by legal' $DOCUMENT_ROOT/.htaccess)
  if [[ ! -z "$check" ]]; then
    sed -i 's/Allow from//g;s/allow from//g;s/Order Allow,Deny//;s/#########################################################################//g;s/# Website has been disabled by legal department #//;s/# Please reach out to legal@inmotionhosting.com #//' $DOCUMENT_ROOT/.htaccess # The above check shouldn't fail but if it does then use sed to unlock .htaccess - this could remove legit Allow/Deny from statements, this is why above a copy of the original htaccess was already made
  fi
}

function add_ips {
  chattr -i $DOCUMENT_ROOT/.htaccess
  if [[ $SUBVERSION -eq 2 ]]; then
    sed -i 's/deny from all//;s/Deny from all//' $DOCUMENT_ROOT/.htaccess
  else
    sed -i 's/Require all denied//;s/require all denied//' $DOCUMENT_ROOT/.htaccess
  fi
  printf "Order Allow,Deny\nAllow from 173.231.218.25\n" > $DOCUMENT_ROOT/new_legal_htaccess # Support IP
  printf "Enter the list of IPs you want to allow\nWhen complete enter the '$' symbol\n"
  read -d '$' list
  for i in $list;do echo "Allow from $i" >> $DOCUMENT_ROOT/new_legal_htaccess;done
  mv -vf $DOCUMENT_ROOT/.htaccess{,LegalDisabled} && cat $DOCUMENT_ROOT/new_legal_htaccess $DOCUMENT_ROOT/.htaccessLegalDisabled > $DOCUMENT_ROOT/.htaccess
  if [[ ! -z "$NGINX_INSTALLED" ]]; then # NGiNX Installed
    ngxconf -r -u $USER
  fi
  chattr +i $DOCUMENT_ROOT/.htaccess
}

function check_domain_locked {
  unset domain_check;domain_check=$(curl -sIA imhLegal --insecure https://$DOMAIN/ | grep -i '403 Forbidden')
  if [[ ! -z "$domain_check" ]]; then
    printf "Domain Locked Successfully!\n$domain_check\n"
  else
    printf "Domain Check Failed!\n$domain_check\nIs the domain live on this server?\n"
  fi
}

function check_domain_unlocked {
  unset domain_check;domain_check=$(curl -sIA imhLegal --insecure https://$DOMAIN/ | grep -i '200 OK')
  if [[ ! -z "$domain_check" ]]; then
    printf "Domain Unlocked Succeeded!\n$domain_check\n"
  else
    printf "Domain Check Failed!\n$domain_check\nIs the domain live on this server?\n"
  fi
}

# Apache Version
apache_version=$(httpd -v | grep version | cut -d '/' -f 2 | awk '{print $1}')
SUBVERSION=$(echo $apache_version | cut -d '.' -f 2)

# NGiNX Installed?
# our nginx binary outputs to stderr NOT stdin, this is why this is being done like this - you can't just redirect stderr
if [[ $(\ls /etc/nginx/nginx.conf 2>/dev/null) ]]; then
  NGINX_INSTALLED=1
fi

# Get Domain
if [[ ! -z "$1" ]]; then
  DOMAIN=$1
else
  read -p 'Domain: ' DOMAIN
fi

grep "^$DOMAIN" /etc/userdomains &> /dev/null #Check if domain is valid
if [[ $? != 0 ]]; then
  printf 'Invalid Domain\nPlease double check spelling or escalate if needed.\n'
  exit 1
fi

if [[ $(/scripts/whoowns $DOMAIN 2> /dev/null) ]]; then #Check if a valid user can be identified
  USER=$(/scripts/whoowns $DOMAIN)
else
  printf "Invalid User\nPlease double check spelling or escalate if needed.\n" #Can add alternative checks here
  exit 1
fi

#Check if domain is the primary domain or an addon
primary_check=$(grep $DOMAIN /etc/trueuserdomains)
if [[ ! -z "$primary_check" ]]; then
  primary_domain=1
else
  primary_domain=0
fi

if [[ $primary_domain -eq 1 ]]; then
  if [[ $(\ls -1 /var/cpanel/userdata/$USER/$DOMAIN 2> /dev/null) ]]; then #Check if userdata exists
    DOCUMENT_ROOT=$(grep documentroot /var/cpanel/userdata/$USER/$DOMAIN | cut -d : -f 2 | tr -d ' ')
  else
    echo 'Invalid User Data\nPlease Escalate\n'
    exit 1
  fi
else
  # Subdomain variant userdata
  pd=$(grep $USER /etc/trueuserdomains | cut -d : -f 1) # Primary domain
  dot_count=$(echo $DOMAIN | grep -o '\.' | wc -l)
  if [[ $dot_count -gt 2 ]]; then
    let 'limit=dot_count-1'
    sd=$(echo "$DOMAIN" | cut -d '.' -f 1-$limit) # Sub-domain short name
  else
    sd=$(echo $DOMAIN | cut -d '.' -f 1) # Sub-domain short name
  fi
  full_domain=$(echo "$sd.$pd") # Make sub-domain for userdata
  DOCUMENT_ROOT=$(grep documentroot /var/cpanel/userdata/$USER/$full_domain | cut -d : -f 2 | tr -d ' ')
fi

#Confirm document root is valid
if [[ ! -d $DOCUMENT_ROOT ]]; then
  echo 'Invalid Document Root\nPlease review Document Root or Escalate if needed\n'
  exit 1
fi

# Run applicable function
unset check && check=$(grep 'disabled by legal' $DOCUMENT_ROOT/.htaccess 2>/dev/null)
# If domain is currently blocked give menu, else lock domain
if [[ ! -z "$check" ]]; then
  echo "Domain Locked"
  printf "Do you want to:\ni) Add IPs to .htaccess\nu) Unlock Domain\nq) Quit\n"
  read menu_choice
  case $menu_choice in
	i)
		add_ips
		echo "IPs added!"
		;;
	I)
		add_ips
		echo "IPs added!"
		;;
	u)
		unlock_domain
		check_domain_unlocked
		;;
	U)
		unlock_domain
		check_domain_unlocked
		;;
	q)
		exit 0
		;;
	Q)
		exit 0
		;;
	*)
		echo "Invalid Choice! Exiting!"
		exit 1
		;;
  esac
else
  lock_domain
  check_domain_locked
fi
